<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);

header("Content-Type: application/json; charset=UTF-8");
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST");

require __DIR__ . '/../private/config.php';

// Pastikan koneksi database aktif
if (!$koneksi) {
    echo json_encode(["status" => "error", "msg" => "Koneksi database gagal"]);
    exit;
}

try {
    $input = json_decode(file_get_contents("php://input"), true);
    if (!$input) throw new Exception("Input tidak valid / bukan JSON");

    $user_id = $input['user_id'] ?? '';
    $base64 = $input['image'] ?? '';
    if (empty($user_id) || empty($base64)) {
        throw new Exception("Data tidak lengkap (user_id / image kosong)");
    }

    // Decode base64
    $data = base64_decode($base64, true);
    if ($data === false || strlen($data) < 50) {
        throw new Exception("Gagal decode base64 atau file terlalu kecil");
    }

    // Deteksi MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mime = finfo_buffer($finfo, $data);
    finfo_close($finfo);
    if (!$mime) throw new Exception("Gagal deteksi tipe file");

    $ext = match ($mime) {
        'image/jpeg' => 'jpg',
        'image/png' => 'png',
        'image/webp' => 'webp',
        default => throw new Exception("Tipe file tidak didukung: " . $mime),
    };

    // Simpan ke file sementara
    $tempFile = sys_get_temp_dir() . '/tmp_' . uniqid() . '.' . $ext;
    file_put_contents($tempFile, $data);

    // Buka gambar
    switch ($ext) {
        case 'jpg': $image = imagecreatefromjpeg($tempFile); break;
        case 'png': $image = imagecreatefrompng($tempFile); break;
        case 'webp': $image = imagecreatefromwebp($tempFile); break;
    }

    if (!$image) throw new Exception("Gagal memuat gambar dari base64");

    // --- 🔹 Resize otomatis seperti WhatsApp (maks. lebar 800px) ---
    $width = imagesx($image);
    $height = imagesy($image);
    $maxWidth = 800;
    if ($width > $maxWidth) {
    $ratio = $height / $width;
    $newWidth = (int)$maxWidth;
    $newHeight = (int)round($newWidth * $ratio);

    $resized = imagecreatetruecolor($newWidth, $newHeight);
    imagecopyresampled($resized, $image, 0, 0, 0, 0, $newWidth, $newHeight, $width, $height);
    imagedestroy($image);
    $image = $resized;
}


    // --- 🔹 Simpan hasil ke folder uploads ---
    $uploadDir = $_SERVER['DOCUMENT_ROOT'] . '/uploads/chat_images/';
    if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

    $fileName = 'IMG_' . $user_id . '_' . uniqid() . '.' . $ext;
    $filePath = $uploadDir . $fileName;

    $quality = 85;
    switch ($ext) {
        case 'jpg': imagejpeg($image, $filePath, $quality); break;
        case 'png': imagepng($image, $filePath, 6); break;
        case 'webp': imagewebp($image, $filePath, $quality); break;
    }
    imagedestroy($image);
    unlink($tempFile);

    $imageUrl = "https://" . $_SERVER['HTTP_HOST'] . "/uploads/chat_images/" . $fileName;

    // --- 🔹 Simpan ke database ---
    $title = $input['title'] ?? 'You';
    $timestamp = (int)($input['timestamp'] ?? (microtime(true) * 1000));
    $is_sender = ($input['is_sender'] ?? true) ? 1 : 0;
    $status = $input['status'] ?? 'sent';
    $type = 'image';

    $stmt = $koneksi->prepare("INSERT INTO chat (user_id, title, message, timestamp, is_sender, status, type)
                               VALUES (?, ?, ?, ?, ?, ?, ?)");
    $stmt->bind_param("sssiiss", $user_id, $title, $imageUrl, $timestamp, $is_sender, $status, $type);
    $stmt->execute();

    // --- 🔹 Kirim notifikasi lewat OneSignal ---
    $q = $koneksi->prepare("SELECT one_id FROM pelanggan WHERE user_id = ?");
    $q->bind_param("s", $user_id);
    $q->execute();
    $res = $q->get_result()->fetch_assoc();
    $player_id = $res['one_id'] ?? null;

    if ($player_id) {
        $notif = [
            "app_id" => ONESIGNAL_APP_ID,
            "include_player_ids" => [$player_id],
            "headings" => ["en" => "Pesan Baru 📷"],
            "contents" => ["en" => "Admin mengirim foto baru"],
            "big_picture" => $imageUrl
        ];

        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, "https://onesignal.com/api/v1/notifications");
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json; charset=utf-8',
            'Authorization: Basic ' . ONESIGNAL_REST_API_KEY
        ]);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($notif));
        curl_exec($ch);
        curl_close($ch);
    }

    echo json_encode([
        "status" => "success",
        "msg" => "Gambar berhasil diunggah & dikirim",
        "data" => [
            "id" => $stmt->insert_id,
            "user_id" => $user_id,
            "title" => $title,
            "image" => $imageUrl,
            "timestamp" => $timestamp,
            "status" => $status,
            "notif_sent" => (bool)$player_id
        ]
    ], JSON_PRETTY_PRINT);

} catch (Exception $e) {
    echo json_encode(["status" => "error", "msg" => $e->getMessage()], JSON_PRETTY_PRINT);
}

$koneksi->close();
?>
